﻿// Only works on ARGB32, RGB24 and Alpha8 textures that are marked readable
using Assets.Scripts;
using System.Threading;
using UnityEngine;

public class TextureScale
{
	public const bool debug = false;

	public enum ImageFilterMode : int
	{
		Nearest = 0,
		Biliner = 1,
		Average = 2
	}

	public static Texture2D ResizeTexture(Texture2D pSource, ImageFilterMode pFilterMode, float xWidth, float xHeight, TextureFormat forcedTextureFormat = 0)
	{
		if (debug)
		{
			ZoLogger.Log(Colors.orange, $"TextureScale.ResizeTexture({pSource.width}x{pSource.height}, {pFilterMode}, {xWidth}, {xHeight}, {forcedTextureFormat})");
		}

		//*** Variables
		int i;

		//*** Get All the source pixels
		Color[] aSourceColor = pSource.GetPixels(0);
		Vector2 vSourceSize = new Vector2(pSource.width, pSource.height);

		//*** Make New
		Texture2D oNewTex = new Texture2D((int)xWidth, (int)xHeight, forcedTextureFormat == 0 ? pSource.format : forcedTextureFormat, false);

		//*** Make destination array
		int xLength = (int)xWidth * (int)xHeight;
		Color[] aColor = new Color[xLength];

		Vector2 vPixelSize = new Vector2(vSourceSize.x / xWidth, vSourceSize.y / xHeight);

		//*** Loop through destination pixels and process
		Vector2 vCenter = new Vector2();
		for (i = 0; i < xLength; i++)
		{

			//*** Figure out x&y
			float xX = (float)i % xWidth;
			float xY = Mathf.Floor((float)i / xWidth);

			//*** Calculate Center
			vCenter.x = (xX / xWidth) * vSourceSize.x;
			vCenter.y = (xY / xHeight) * vSourceSize.y;

			//*** Do Based on mode
			//*** Nearest neighbour (testing)
			if (pFilterMode == ImageFilterMode.Nearest)
			{

				//*** Nearest neighbour (testing)
				vCenter.x = Mathf.Round(vCenter.x);
				vCenter.y = Mathf.Round(vCenter.y);

				//*** Calculate source index
				int xSourceIndex = (int)((vCenter.y * vSourceSize.x) + vCenter.x);

				//*** Copy Pixel
				aColor[i] = aSourceColor[xSourceIndex];
			}

			//*** Bilinear
			else if (pFilterMode == ImageFilterMode.Biliner)
			{

				//*** Get Ratios
				float xRatioX = vCenter.x - Mathf.Floor(vCenter.x);
				float xRatioY = vCenter.y - Mathf.Floor(vCenter.y);

				//*** Get Pixel index's
				int xIndexTL = (int)((Mathf.Floor(vCenter.y) * vSourceSize.x) + Mathf.Floor(vCenter.x));
				int xIndexTR = (int)((Mathf.Floor(vCenter.y) * vSourceSize.x) + Mathf.Ceil(vCenter.x));
				int xIndexBL = (int)((Mathf.Ceil(vCenter.y) * vSourceSize.x) + Mathf.Floor(vCenter.x));
				int xIndexBR = (int)((Mathf.Ceil(vCenter.y) * vSourceSize.x) + Mathf.Ceil(vCenter.x));

				//*** Calculate Color
				aColor[i] = Color.Lerp(
					Color.Lerp(aSourceColor[xIndexTL], aSourceColor[xIndexTR], xRatioX),
					Color.Lerp(aSourceColor[xIndexBL], aSourceColor[xIndexBR], xRatioX),
					xRatioY
				);
			}

			//*** Average
			else if (pFilterMode == ImageFilterMode.Average)
			{

				//*** Calculate grid around point
				int xXFrom = (int)Mathf.Max(Mathf.Floor(vCenter.x - (vPixelSize.x * 0.5f)), 0);
				int xXTo = (int)Mathf.Min(Mathf.Ceil(vCenter.x + (vPixelSize.x * 0.5f)), vSourceSize.x);
				int xYFrom = (int)Mathf.Max(Mathf.Floor(vCenter.y - (vPixelSize.y * 0.5f)), 0);
				int xYTo = (int)Mathf.Min(Mathf.Ceil(vCenter.y + (vPixelSize.y * 0.5f)), vSourceSize.y);

				//*** Loop and accumulate
				Color oColorTemp = new Color();
				float xGridCount = 0;
				float xGridCounta = 0;
				for (int iy = xYFrom; iy < xYTo; iy++)
				{
					for (int ix = xXFrom; ix < xXTo; ix++)
					{
						//Color pixel = GetColor(ref aSourceColor, ix, iy);
						Color pixel = aSourceColor[(int)(((float)iy * vSourceSize.x) + ix)];

						//*** Get Color
						oColorTemp += pixel * pixel.a;

						//*** Sum
						xGridCount += pixel.a;
						xGridCounta++;
					}
				}

				//*** Average Color				

				aColor[i] = oColorTemp / (float)xGridCount;
				aColor[i].a = xGridCount / xGridCounta;
			}
		}

		//*** Set Pixels
		oNewTex.SetPixels(aColor);
		oNewTex.Apply();

		//*** Return
		return oNewTex;
	}

	static int lineSize;
	static Rect sourceDeviation;
	static Vector2 deviation;
	static float curXWidth;
	static float curXHeight;
	static float sourceRatio;
	static float curZoom = 1.0f;


	static Color GetColor(ref Color[] aSourceColor, int x, int y)
	{
		x = (int)((x - sourceDeviation.x) * sourceDeviation.width * curZoom + sourceDeviation.x);
		y = (int)((y - sourceDeviation.y) * sourceDeviation.height * curZoom + sourceDeviation.y);

		//if (xSourceIndex < 0 || xSourceIndex >= aSourceColor.Length)

		if (x < 0 || x >= curXWidth || y < 0 || y >= curXHeight)
		{
			return new Color(0f, 0f, 0f, 0f);
		}
		else
		{
			int xSourceIndex = (int)((y * lineSize) + x);
			return aSourceColor[xSourceIndex];
		}
	}

	public static Texture2D ResizeTextureKeepRatio(Texture2D pSource, ImageFilterMode pFilterMode, float xWidth, float xHeight, float zoom = 1.0f, TextureFormat forcedTextureFormat = 0)
	{
		if (debug)
		{
			ZoLogger.Log(Colors.orange, $"TextureScale.ResizeTextureKeepRatio({pSource.width}x{pSource.height}, {pFilterMode}, {xWidth}, {xHeight}, {zoom}, {forcedTextureFormat})");
		}

		curXWidth = pSource.width;
		curXHeight = pSource.height;
		curZoom = zoom;

		//*** Get All the source pixels
		Color[] aSourceColor = pSource.GetPixels(0);
		Vector2 vSourceSize = new(pSource.width, pSource.height);

		sourceRatio = (float)pSource.width / (float)pSource.height;
		float destRatio = xWidth / xHeight;

		//sourceDeviation = new Rect(	(float)pSource.width * 0.5f,
		//							(float)pSource.height * 0.5f,
		//							(float)pSource.height * destRatio / (float)pSource.width,
		//							(float)pSource.width / destRatio / (float)pSource.height);
		if (sourceRatio > destRatio)
		{
			Debug.Log("Horizontal " + sourceRatio);
			sourceDeviation = new Rect(pSource.width * 0.5f, (float)pSource.height * 0.5f, 1.0f, (float)pSource.width / destRatio / (float)pSource.height);
		}
		else
		{
			Debug.Log("Vertical");
			sourceDeviation = new Rect(pSource.width * 0.5f, (float)pSource.height * 0.5f, (float)pSource.height * destRatio / (float)pSource.width, 1.0f);
		}

		//*** Make New
		Texture2D oNewTex = new Texture2D((int)xWidth, (int)xHeight, forcedTextureFormat == 0 ? pSource.format : forcedTextureFormat, false);

		//*** Make destination array
		int xLength = (int)xWidth * (int)xHeight;
		Color[] aColor = new Color[xLength];

		Vector2 vPixelSize = new Vector2(vSourceSize.x / xWidth, vSourceSize.y / xHeight);

		lineSize = (int)pSource.width;

		//*** Loop through destination pixels and process
		Vector2 vCenter = new Vector2();
		for (int i = 0; i < xWidth; i++)
		{
			for (int j = 0; j < xHeight; j++)
			{
				//*** Figure out x&y
				float xX = (float)i;
				float xY = Mathf.Floor((float)j);

				//*** Calculate Center in source Image
				vCenter.x = (xX / xWidth) * vSourceSize.x;
				vCenter.y = (xY / xHeight) * vSourceSize.y;

				//*** Do Based on mode
				//*** Nearest neighbour (testing)
				if (pFilterMode == ImageFilterMode.Nearest)
				{

					//*** Nearest neighbour (testing)
					vCenter.x = Mathf.Round(vCenter.x);
					vCenter.y = Mathf.Round(vCenter.y);

					aColor[i + j * (int)xWidth] = GetColor(ref aSourceColor, (int)vCenter.x, (int)vCenter.y);

					////*** Calculate source index
					//int xSourceIndex = (int)((vCenter.y * vSourceSize.x) + vCenter.x);

					////*** Copy Pixel
					//aColor[i + j * (int)xWidth] = aSourceColor[xSourceIndex];
				}

				//*** Bilinear
				else if (pFilterMode == ImageFilterMode.Biliner)
				{

					//*** Get Ratios
					float xRatioX = vCenter.x - Mathf.Floor(vCenter.x);
					float xRatioY = vCenter.y - Mathf.Floor(vCenter.y);


					aColor[i + j * (int)xWidth] = Color.Lerp(
						Color.Lerp(	GetColor(ref aSourceColor, (int)vCenter.x, (int)vCenter.y),
									GetColor(ref aSourceColor, (int)Mathf.Ceil(vCenter.x), (int)vCenter.y),
									xRatioX),
						Color.Lerp(GetColor(ref aSourceColor, (int)vCenter.x, (int)Mathf.Ceil(vCenter.y)),
									GetColor(ref aSourceColor, (int)Mathf.Ceil(vCenter.x), (int)Mathf.Ceil(vCenter.x)),
									xRatioX),
						xRatioY
					);
					
					////*** Get Pixel index's
					//int xIndexTL = (int)((Mathf.Floor(vCenter.y) * vSourceSize.x) + Mathf.Floor(vCenter.x));
					//int xIndexTR = (int)((Mathf.Floor(vCenter.y) * vSourceSize.x) + Mathf.Ceil(vCenter.x));
					//int xIndexBL = (int)((Mathf.Ceil(vCenter.y) * vSourceSize.x) + Mathf.Floor(vCenter.x));
					//int xIndexBR = (int)((Mathf.Ceil(vCenter.x) * vSourceSize.x) + Mathf.Ceil(vCenter.x));

					////*** Calculate Color
					//aColor[i + j * (int)xWidth] = Color.Lerp(
					//	Color.Lerp(aSourceColor[xIndexTL], aSourceColor[xIndexTR], xRatioX),
					//	Color.Lerp(aSourceColor[xIndexBL], aSourceColor[xIndexBR], xRatioX),
					//	xRatioY
					//);
				}

				//*** Average
				else if (pFilterMode == ImageFilterMode.Average)
				{

					//*** Calculate grid around point
					int xXFrom = (int)Mathf.Max(Mathf.Floor(vCenter.x - (vPixelSize.x * 0.5f)), 0);
					int xXTo = (int)Mathf.Min(Mathf.Ceil(vCenter.x + (vPixelSize.x * 0.5f)), vSourceSize.x);
					int xYFrom = (int)Mathf.Max(Mathf.Floor(vCenter.y - (vPixelSize.y * 0.5f)), 0);
					int xYTo = (int)Mathf.Min(Mathf.Ceil(vCenter.y + (vPixelSize.y * 0.5f)), vSourceSize.y);

					//vCenter.x = Mathf.Round(vCenter.x);
					//vCenter.y = Mathf.Round(vCenter.y);
				
					//*** Loop and accumulate
					//Vector4 oColorTotal = new Vector4();
					Color oColorTemp = new Color();
					float xGridCount = 0;
					float xGridCounta = 0;
					for (int iy = xYFrom; iy < xYTo; iy++)
					{
						for (int ix = xXFrom; ix < xXTo; ix++)
						{
							Color pixel = GetColor(ref aSourceColor, ix, iy);
							
							//*** Get Color
							oColorTemp += pixel * pixel.a;

							//*** Sum
							xGridCount += pixel.a;
							xGridCounta++;
						}
					}

					//*** Average Color				

					aColor[i + j * (int)xWidth] = oColorTemp / (float)xGridCount;
					aColor[i + j * (int)xWidth].a = xGridCount / xGridCounta;
				}
			}
		}

		//*** Set Pixels
		oNewTex.SetPixels(aColor);
		oNewTex.Apply();

		//*** Return
		return oNewTex;
	}

	public static Texture2D ResizeTextureKeepRatio(Texture2D pSource, ImageFilterMode pFilterMode, float xWidth, float xHeight, Rect zone, TextureFormat forcedTextureFormat = 0)
	{
		if (debug)
		{
			ZoLogger.Log(Colors.orange, $"TextureScale.ResizeTextureKeepRatio({pSource.width}x{pSource.height}, {pFilterMode}, {xWidth}, {xHeight}, {zone}, {forcedTextureFormat})");
		}

		xWidth = Mathf.RoundToInt(xWidth);
		xHeight = Mathf.RoundToInt(xHeight);

		curXWidth = pSource.width;
		curXHeight = pSource.height;

		//*** Get All the source pixels
		Color[] aSourceColor = pSource.GetPixels(0);

		//*** Make New
		Texture2D oNewTex = new Texture2D((int)xWidth, (int)xHeight, forcedTextureFormat == 0 ? pSource.format : forcedTextureFormat, false);

		int originStartX = (int)(pSource.width * zone.x);
		int originStartY = (int)(pSource.height * zone.y);
		int originEndX = (int)(pSource.width * zone.xMax);
		int originEndY = (int)(pSource.height * zone.yMax);

		Vector2 vSourceSize = new Vector2(pSource.width * zone.width, pSource.height * zone.height);

		float sourceRatio = pSource.width / pSource.height;
		float rectRatio = vSourceSize.x / vSourceSize.y;
		float destRatio = xWidth / xHeight;

		if (rectRatio > destRatio)
		{
			//deviation = new Vector2(1.0f, pSource.width / destRatio / pSource.height);
			sourceDeviation = new Rect(0.0f, xHeight * 0.5f, 1.0f, pSource.width / destRatio / pSource.height);
		}
		else
		{
			//deviation = new Vector2(pSource.height * destRatio / pSource.width, 1.0f);
			sourceDeviation = new Rect(xWidth * 0.5f, 0.0f, pSource.height * destRatio / pSource.width, 1.0f);
		}

		//*** Make destination array
		int xLength = (int)xWidth * (int)xHeight;
		Color[] aColor = new Color[xLength];

		Vector2 vPixelSize = new Vector2(vSourceSize.x / xWidth, vSourceSize.y / xHeight);
		
		lineSize = (int)pSource.width;

		//*** Loop through destination pixels and process
		Vector2 vCenter = new Vector2();
		for (int i = 0; i < xWidth; i++)
			for (int j = 0; j < xHeight; j++)
			{

				//*** Figure out x&y
				float xX = (float)i;
				float xY = Mathf.Floor((float)j);

				//*** Calculate Center
				vCenter.x = (xX / xWidth) * vSourceSize.x + originStartX;
				vCenter.y = (xY / xHeight) * vSourceSize.y + originStartY;

				//*** Do Based on mode
				//*** Nearest neighbour (testing)
				if (pFilterMode == ImageFilterMode.Nearest)
				{

					//*** Nearest neighbour (testing)
					vCenter.x = Mathf.Round(vCenter.x);
					vCenter.y = Mathf.Round(vCenter.y);

					//*** Calculate source index
					int xSourceIndex = (int)((vCenter.y * pSource.width) + vCenter.x);

					//*** Copy Pixel
					aColor[i + j * (int)xWidth] = aSourceColor[xSourceIndex];
				}

				//*** Bilinear
				else if (pFilterMode == ImageFilterMode.Biliner)
				{

					//*** Get Ratios
					float xRatioX = vCenter.x - Mathf.Floor(vCenter.x);
					float xRatioY = vCenter.y - Mathf.Floor(vCenter.y);

					//*** Get Pixel index's
					int xIndexTL = (int)((Mathf.Floor(vCenter.y) * pSource.width) + Mathf.Floor(vCenter.x));
					int xIndexTR = (int)((Mathf.Floor(vCenter.y) * pSource.width) + Mathf.Ceil(vCenter.x));
					int xIndexBL = (int)((Mathf.Ceil(vCenter.y) * pSource.width) + Mathf.Floor(vCenter.x));
					int xIndexBR = (int)((Mathf.Ceil(vCenter.y) * pSource.width) + Mathf.Ceil(vCenter.x));

					//*** Calculate Color
					aColor[i + j * (int)xWidth] = Color.Lerp(
						Color.Lerp(aSourceColor[xIndexTL], aSourceColor[xIndexTR], xRatioX),
						Color.Lerp(aSourceColor[xIndexBL], aSourceColor[xIndexBR], xRatioX),
						xRatioY
					);
				}

				//*** Average
				else if (pFilterMode == ImageFilterMode.Average)
				{

					//*** Calculate grid around point
					int xXFrom = (int)Mathf.Max(Mathf.Floor(vCenter.x - (vPixelSize.x * 0.5f)), 0);
					int xXTo = (int)Mathf.Min(Mathf.Ceil(vCenter.x + (vPixelSize.x * 0.5f)), pSource.width);
					int xYFrom = (int)Mathf.Max(Mathf.Floor(vCenter.y - (vPixelSize.y * 0.5f)), 0);
					int xYTo = (int)Mathf.Min(Mathf.Ceil(vCenter.y + (vPixelSize.y * 0.5f)), pSource.height);

					//*** Loop and accumulate
					Color oColorTemp = new Color();
					float xGridCount = 0;
					float xGridCounta = 0;
					for (int iy = xYFrom; iy < xYTo; iy++)
					{
						for (int ix = xXFrom; ix < xXTo; ix++)
						{
							Color pixel = aSourceColor[(int)(((float)iy * pSource.width) + ix)];

							//*** Get Color
							oColorTemp += pixel * pixel.a;

							//*** Sum
							xGridCount += pixel.a;
							xGridCounta++;
						}
					}

					//*** Average Color				

					aColor[i + j * (int)xWidth] = oColorTemp / (float)xGridCount;
					aColor[i + j * (int)xWidth].a = xGridCount / xGridCounta;
				}
			}

		//*** Set Pixels
		oNewTex.SetPixels(aColor);
		oNewTex.Apply();

		//*** Return
		return oNewTex;
	}

	public static Texture2D ResizeTexture(Texture2D pSource, ImageFilterMode pFilterMode, float xWidth, float xHeight, Rect zone, TextureFormat forcedTextureFormat = 0)
	{
		if (debug)
		{
			ZoLogger.Log(Colors.orange, $"TextureScale.ResizeTexture({pSource.width}x{pSource.height}, {pFilterMode}, {xWidth}, {xHeight}, {zone}, {forcedTextureFormat})");
		}

		xWidth = Mathf.RoundToInt(xWidth);
		xHeight = Mathf.RoundToInt(xHeight);

		//*** Get All the source pixels
		Color[] aSourceColor = pSource.GetPixels(0);

		//*** Make New
		Texture2D oNewTex = new Texture2D((int)xWidth, (int)xHeight, forcedTextureFormat == 0 ? pSource.format : forcedTextureFormat, false);

		int originStartX = (int)(pSource.width * zone.x);
		int originStartY = (int)(pSource.height * zone.y);
		int originEndX = (int)(pSource.width * zone.xMax);
		int originEndY = (int)(pSource.height * zone.yMax);

		Vector2 vSourceSize = new Vector2(pSource.width * zone.width, pSource.height * zone.height);

		//*** Make destination array
		int xLength = (int)xWidth * (int)xHeight;
		Color[] aColor = new Color[xLength];

		Vector2 vPixelSize = new Vector2(vSourceSize.x / xWidth, vSourceSize.y / xHeight);

		//*** Loop through destination pixels and process
		Vector2 vCenter = new Vector2();
		for (int i = 0; i < xWidth; i++)
			for (int j = 0; j < xHeight; j++)
			{

				//*** Figure out x&y
				float xX = (float)i;
				float xY = Mathf.Floor((float)j);

				//*** Calculate Center
				vCenter.x = (xX / xWidth) * vSourceSize.x + originStartX;
				vCenter.y = (xY / xHeight) * vSourceSize.y + originStartY;

				//*** Do Based on mode
				//*** Nearest neighbour (testing)
				if (pFilterMode == ImageFilterMode.Nearest)
				{

					//*** Nearest neighbour (testing)
					vCenter.x = Mathf.Round(vCenter.x);
					vCenter.y = Mathf.Round(vCenter.y);

					//*** Calculate source index
					int xSourceIndex = (int)((vCenter.y * pSource.width) + vCenter.x);

					//*** Copy Pixel
					aColor[i + j * (int)xWidth] = aSourceColor[xSourceIndex];
				}

				//*** Bilinear
				else if (pFilterMode == ImageFilterMode.Biliner)
				{

					//*** Get Ratios
					float xRatioX = vCenter.x - Mathf.Floor(vCenter.x);
					float xRatioY = vCenter.y - Mathf.Floor(vCenter.y);

					//*** Get Pixel index's
					int xIndexTL = (int)((Mathf.Floor(vCenter.y) * pSource.width) + Mathf.Floor(vCenter.x));
					int xIndexTR = (int)((Mathf.Floor(vCenter.y) * pSource.width) + Mathf.Ceil(vCenter.x));
					int xIndexBL = (int)((Mathf.Ceil(vCenter.y) * pSource.width) + Mathf.Floor(vCenter.x));
					int xIndexBR = (int)((Mathf.Ceil(vCenter.y) * pSource.width) + Mathf.Ceil(vCenter.x));

					//*** Calculate Color
					aColor[i + j * (int)xWidth] = Color.Lerp(
						Color.Lerp(aSourceColor[xIndexTL], aSourceColor[xIndexTR], xRatioX),
						Color.Lerp(aSourceColor[xIndexBL], aSourceColor[xIndexBR], xRatioX),
						xRatioY
					);
				}

				//*** Average
				else if (pFilterMode == ImageFilterMode.Average)
				{

					//*** Calculate grid around point
					int xXFrom = (int)Mathf.Max(Mathf.Floor(vCenter.x - (vPixelSize.x * 0.5f)), 0);
					int xXTo = (int)Mathf.Min(Mathf.Ceil(vCenter.x + (vPixelSize.x * 0.5f)), pSource.width);
					int xYFrom = (int)Mathf.Max(Mathf.Floor(vCenter.y - (vPixelSize.y * 0.5f)), 0);
					int xYTo = (int)Mathf.Min(Mathf.Ceil(vCenter.y + (vPixelSize.y * 0.5f)), pSource.height);

					//*** Loop and accumulate
					Color oColorTemp = new Color();
					float xGridCount = 0;
					float xGridCounta = 0;
					for (int iy = xYFrom; iy < xYTo; iy++)
					{
						for (int ix = xXFrom; ix < xXTo; ix++)
						{
							Color pixel = aSourceColor[(int)(((float)iy * pSource.width) + ix)]; 

							//*** Get Color
							oColorTemp += pixel * pixel.a;

							//*** Sum
							xGridCount += pixel.a;
							xGridCounta++;
						}
					}

					//*** Average Color				

					aColor[i + j * (int)xWidth] = oColorTemp / (float)xGridCount;
					aColor[i + j * (int)xWidth].a = xGridCount / xGridCounta;
				}
			}

		//*** Set Pixels
		oNewTex.SetPixels(aColor);
		oNewTex.Apply();

		//*** Return
		return oNewTex;
	}

	public static Texture2D ChangeFormat(Texture2D pSource, TextureFormat forcedTextureFormat)
	{
		//*** Get All the source pixels
		Color32[] aSourceColor = pSource.GetPixels32(0);

		//*** Make New
		Texture2D oNewTex = new Texture2D(pSource.width, pSource.height, forcedTextureFormat, false);

		//*** Set Pixels
		oNewTex.SetPixels32(aSourceColor);
		oNewTex.Apply();

		//*** Return
		return oNewTex;
	}

}
